# GBA Song Editor
#
# Copyright 2010-2011 Karl A. Knechtel.
#
# Main program. A basic tool for working with GBA ROM audio and MIDI files.
#
# Licensed under the Generic Non-Commercial Copyleft Software License,
# Version 1.1 (hereafter "Licence"). You may not use this file except
# in the ways outlined in the Licence, which you should have received
# along with this file.
#
# Unless required by applicable law or agreed to in writing, software 
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
# implied. See the License for the specific language governing
# permissions and limitations under the License.


from cli import command, display, main, NO_WARRANTY, PEBKAC
from util import State, Chunk, InstrumentMap, save_chunk, rip_tracks, process_song, do_burning, do_dump
from midi import do_conversion


VERSION = (0, 46)


def integral(arg, name):
	try: return int(arg, 0)
	except: raise PEBKAC, "%s must be an integer in base 16 (preceded by 0x) or 8 (preceded by 0) or 10." % name


def tick_count(kwargs, name, desc):
	if name not in kwargs: return None
	x, c, y = kwargs[name].partition(':')
	if y == '': y = '0'
	try: return int(x) * 24 + int(y)
	except: raise PEBKAC, "%s must be an integer in base 10, or a tick count of the form <beats>:<ticks>, with each an integer in base 10." % desc


def read_file(base, *extensions):
	for extension in extensions:
		try:
			filename = '%s.%s' % (base, extension)
			with file(filename, 'rb') as f: return filename, f.read()
		except:
			pass # try the next one.

	raise PEBKAC("File not found.")


@command()
def open(state, filename):
	"""Loads a ROM from disk.

	The file is copied into memory and released immediately so that other
	programs can use it. You can revert changes by re-opening the ROM without
	saving."""

	state.ROM_file, data = read_file(filename, 'gba')
	state.ROM = Chunk(data)
	display("Loaded ROM from disk.")


@command()
def close(state, filename):
	"""Saves a ROM to disk and "close" it (make it unavailable for further work).

	If the file already exists, it will be overwritten; otherwise, a new file
	will be created. The file is written and released immediately so that
	other programs can use it. You can save work on a ROM and continue working
	with it by closing it and then re-opening it."""

	if not filename: raise PEBKAC("A file name is required.")
	save_chunk(filename + '.gba', state.ROM)
	state.reset_ROM()
	display("Saved ROM to disk.")


@command()
def rip(state, offset):
	"""Extracts the song at the specified location from the current ROM.
	It can then be 'save'd or 'burn'ed."""

	source = state.ROM
	if not source: raise PEBKAC("No ROM is open.")
	offset = integral(offset, 'Offset')
	try:
		track_count = source[offset]
		instrument_map = InstrumentMap(source, source.read_int(offset + 4))

		track_data, track_starts, bad_tracks = rip_tracks(
			source, offset, instrument_map, track_count
		)
	except IndexError:
		raise PEBKAC("Invalid song offset.")

	for bt in bad_tracks:
		display("Track contained a currently unsupported instrument type '%s'. Skipped." % bt)

	if not track_starts:
		raise PEBKAC("No valid tracks were found. Song not ripped.")
	display("Ripped %d of %d tracks." % (len(track_starts), source[offset]))

	state.song_data = process_song(track_data, track_starts, instrument_map)
	state.song_source = (offset, state.ROM_file)
	display("Song successfully ripped.")


@command() # handle = 'offset for pointer', destination = 'offset for writing data'
def burn(state, handle, destination = None):
	"""Writes the song into the current ROM and set a pointer to it.

	If a destination offset is provided, the song data overwrites any
	existing data at that location. Otherwise the data is appended to
	the ROM. In either case, a pointer (including the GBA ROM offset)
	is written at the provided offset, pointing to the song data."""

	target = state.ROM
	if target == None: raise PEBKAC("No ROM is open.")

	data = state.song_data
	handle = integral(handle, "Offset for pointer")
	destination = len(target) if destination == None else integral(destination, "Destination offset")

	if handle >= destination and handle < destination + len(data):
		raise PEBKAC("Invalid burn position (handle would be within the data).")

	do_burning(target, handle, data, destination)
	display("Burning successful.")


@command('l!loop:loop point in ticks', 'b!begin:beginning point in ticks', 'e!end:end point in ticks')
def convert(state, instrument_map_offset, filename, **kwargs):
	"""Loads a song from a MIDI file. The extension may be '.mid' or '.midi'.
	You need to specify an offset into the instrument map in the target ROM,
	since the MIDI does not contain any mapping - it just assumes a default
	mapping of instruments and default samples (and there may not be anything in
	the ROM which is at all like what is needed, unless you put it there
	yourself).

	If a loop point is provided, it should be a number of ticks. The song will
	loop at the specified number of GBA time ticks past the beginning (all tracks
	will be kept in sync automatically). There are 24 GBA ticks to a beat, so
	e.g. to skip the first 4 bars of the song when looping, use 'loop 384'
	(24 ticks/beat * 4 beats/bar * 4 bars)."""

	name, data = read_file(filename, 'mid', 'midi')
	state.song_data, status = do_conversion(
		data,
		integral(instrument_map_offset, "Offset of instrument map"),
		tick_count(kwargs, 'loop', "Loop point"),
		tick_count(kwargs, 'begin', "Begin point"),
		tick_count(kwargs, 'end', "End point")
	)
	state.song_source = (name,)
	for line in status: display(line)


@command()
def load(state, filename):
	"""Loads a song from a binary dump. Songs saved by the program use the extension '.bin'."""

	source, data = read_file(filename, 'bin')
	state.song_source, state.song_data = (source,), Chunk(data)
	display("Loaded song from file.")


@command()
def save(state, filename):
	"""Saves a binary dump of a song. Songs saved by the program use the extension '.bin'."""
	if state.song_data == None:
		raise PEBKAC("There is no song data to dump.")
	save_chunk(filename + '.bin', state.song_data)
	display("Dumped song to a file. (It is still available for burning.)")


@command()
def info(state):
	"""Shows information about the currently open ROM and song (if applicable)
	and some statistics (e.g. how much space is required to insert the song)."""

	if state.ROM_file == None:
		display('Current ROM: None')
	else:
		display('Current ROM: loaded from file %s' % state.ROM_file)
	if state.song_source == None:
		display('Current song: None')
	else:
		if len(state.song_source) == 1:
			display('Current song: loaded from file %s' % state.song_source)
		else:
			display('Current song: loaded from 0x%08X in ROM %s' % state.song_source)
		display('Insertion, including original samples and own instrument map,')
		size = len(state.song_data)
		display('will require 0x%X (%d) contiguous bytes.' % (size, size))


@command('f!format:output format ("aiff" or "wav")')
def dump(state, folder, start, end, **kwargs):
	"""Dump all used samples from the current ROM, using the provided
	song array locations to find samples.

	Each song that the array refers to has its instrument map looked up,
	and then the samples for instruments in each map are looked up. Once
	the entire listing is complete, the pointers to samples are sorted in
	ascending order, and the samples are dumped to the specified folder."""

	f = kwargs.get('format', 'aiff')
	do_dump(state.ROM, integral(start, 'start'), integral(end, 'end'), folder, f)


if __name__ == '__main__':
	GETTING_STARTED = """\n\nTry 'help' to see a list of commands and a basic
	overview of how to use the program."""
	main('Song Editor', VERSION, State(), NO_WARRANTY + GETTING_STARTED)
